package ipcrypt

import (
	"errors"
)

// AES round constants
var rcon = [10]byte{
	0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36,
}

// Precomputed tables for multiplication by 2 and 3 in GF(2^8)
var (
	gmul2 = [256]byte{
		0x00, 0x02, 0x04, 0x06, 0x08, 0x0a, 0x0c, 0x0e, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1a, 0x1c, 0x1e,
		0x20, 0x22, 0x24, 0x26, 0x28, 0x2a, 0x2c, 0x2e, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3a, 0x3c, 0x3e,
		0x40, 0x42, 0x44, 0x46, 0x48, 0x4a, 0x4c, 0x4e, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5a, 0x5c, 0x5e,
		0x60, 0x62, 0x64, 0x66, 0x68, 0x6a, 0x6c, 0x6e, 0x70, 0x72, 0x74, 0x76, 0x78, 0x7a, 0x7c, 0x7e,
		0x80, 0x82, 0x84, 0x86, 0x88, 0x8a, 0x8c, 0x8e, 0x90, 0x92, 0x94, 0x96, 0x98, 0x9a, 0x9c, 0x9e,
		0xa0, 0xa2, 0xa4, 0xa6, 0xa8, 0xaa, 0xac, 0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc, 0xbe,
		0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc, 0xde,
		0xe0, 0xe2, 0xe4, 0xe6, 0xe8, 0xea, 0xec, 0xee, 0xf0, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfc, 0xfe,
		0x1b, 0x19, 0x1f, 0x1d, 0x13, 0x11, 0x17, 0x15, 0x0b, 0x09, 0x0f, 0x0d, 0x03, 0x01, 0x07, 0x05,
		0x3b, 0x39, 0x3f, 0x3d, 0x33, 0x31, 0x37, 0x35, 0x2b, 0x29, 0x2f, 0x2d, 0x23, 0x21, 0x27, 0x25,
		0x5b, 0x59, 0x5f, 0x5d, 0x53, 0x51, 0x57, 0x55, 0x4b, 0x49, 0x4f, 0x4d, 0x43, 0x41, 0x47, 0x45,
		0x7b, 0x79, 0x7f, 0x7d, 0x73, 0x71, 0x77, 0x75, 0x6b, 0x69, 0x6f, 0x6d, 0x63, 0x61, 0x67, 0x65,
		0x9b, 0x99, 0x9f, 0x9d, 0x93, 0x91, 0x97, 0x95, 0x8b, 0x89, 0x8f, 0x8d, 0x83, 0x81, 0x87, 0x85,
		0xbb, 0xb9, 0xbf, 0xbd, 0xb3, 0xb1, 0xb7, 0xb5, 0xab, 0xa9, 0xaf, 0xad, 0xa3, 0xa1, 0xa7, 0xa5,
		0xdb, 0xd9, 0xdf, 0xdd, 0xd3, 0xd1, 0xd7, 0xd5, 0xcb, 0xc9, 0xcf, 0xcd, 0xc3, 0xc1, 0xc7, 0xc5,
		0xfb, 0xf9, 0xff, 0xfd, 0xf3, 0xf1, 0xf7, 0xf5, 0xeb, 0xe9, 0xef, 0xed, 0xe3, 0xe1, 0xe7, 0xe5,
	}
	gmul3 = [256]byte{
		0x00, 0x03, 0x06, 0x05, 0x0c, 0x0f, 0x0a, 0x09, 0x18, 0x1b, 0x1e, 0x1d, 0x14, 0x17, 0x12, 0x11,
		0x30, 0x33, 0x36, 0x35, 0x3c, 0x3f, 0x3a, 0x39, 0x28, 0x2b, 0x2e, 0x2d, 0x24, 0x27, 0x22, 0x21,
		0x60, 0x63, 0x66, 0x65, 0x6c, 0x6f, 0x6a, 0x69, 0x78, 0x7b, 0x7e, 0x7d, 0x74, 0x77, 0x72, 0x71,
		0x50, 0x53, 0x56, 0x55, 0x5c, 0x5f, 0x5a, 0x59, 0x48, 0x4b, 0x4e, 0x4d, 0x44, 0x47, 0x42, 0x41,
		0xc0, 0xc3, 0xc6, 0xc5, 0xcc, 0xcf, 0xca, 0xc9, 0xd8, 0xdb, 0xde, 0xdd, 0xd4, 0xd7, 0xd2, 0xd1,
		0xf0, 0xf3, 0xf6, 0xf5, 0xfc, 0xff, 0xfa, 0xf9, 0xe8, 0xeb, 0xee, 0xed, 0xe4, 0xe7, 0xe2, 0xe1,
		0xa0, 0xa3, 0xa6, 0xa5, 0xac, 0xaf, 0xaa, 0xa9, 0xb8, 0xbb, 0xbe, 0xbd, 0xb4, 0xb7, 0xb2, 0xb1,
		0x90, 0x93, 0x96, 0x95, 0x9c, 0x9f, 0x9a, 0x99, 0x88, 0x8b, 0x8e, 0x8d, 0x84, 0x87, 0x82, 0x81,
		0x9b, 0x98, 0x9d, 0x9e, 0x97, 0x94, 0x91, 0x92, 0x83, 0x80, 0x85, 0x86, 0x8f, 0x8c, 0x89, 0x8a,
		0xab, 0xa8, 0xad, 0xae, 0xa7, 0xa4, 0xa1, 0xa2, 0xb3, 0xb0, 0xb5, 0xb6, 0xbf, 0xbc, 0xb9, 0xba,
		0xfb, 0xf8, 0xfd, 0xfe, 0xf7, 0xf4, 0xf1, 0xf2, 0xe3, 0xe0, 0xe5, 0xe6, 0xef, 0xec, 0xe9, 0xea,
		0xcb, 0xc8, 0xcd, 0xce, 0xc7, 0xc4, 0xc1, 0xc2, 0xd3, 0xd0, 0xd5, 0xd6, 0xdf, 0xdc, 0xd9, 0xda,
		0x5b, 0x58, 0x5d, 0x5e, 0x57, 0x54, 0x51, 0x52, 0x43, 0x40, 0x45, 0x46, 0x4f, 0x4c, 0x49, 0x4a,
		0x6b, 0x68, 0x6d, 0x6e, 0x67, 0x64, 0x61, 0x62, 0x73, 0x70, 0x75, 0x76, 0x7f, 0x7c, 0x79, 0x7a,
		0x3b, 0x38, 0x3d, 0x3e, 0x37, 0x34, 0x31, 0x32, 0x23, 0x20, 0x25, 0x26, 0x2f, 0x2c, 0x29, 0x2a,
		0x0b, 0x08, 0x0d, 0x0e, 0x07, 0x04, 0x01, 0x02, 0x13, 0x10, 0x15, 0x16, 0x1f, 0x1c, 0x19, 0x1a,
	}
)

// AES S-box
var sbox = [256]byte{
	0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16,
}

// AES inverse S-box
var invSbox = [256]byte{
	0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
	0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
	0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
	0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
	0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
	0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
	0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
	0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
	0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
	0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
	0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
	0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
	0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
	0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
	0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
	0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d,
}

// gmul performs multiplication in GF(2^8)
func gmul(a, b byte) byte {
	var p byte
	for i := 0; i < 8; i++ {
		if (b & 1) != 0 {
			p ^= a
		}
		highBitSet := (a & 0x80) != 0
		a <<= 1
		if highBitSet {
			a ^= 0x1b
		}
		b >>= 1
	}
	return p
}

// shiftRows performs the ShiftRows operation on the state
func shiftRows(state []byte) {
	// Row 0: no shift
	// Row 1: shift left by 1
	state[1], state[5], state[9], state[13] = state[5], state[9], state[13], state[1]
	// Row 2: shift left by 2
	state[2], state[6], state[10], state[14] = state[10], state[14], state[2], state[6]
	// Row 3: shift left by 3
	state[3], state[7], state[11], state[15] = state[15], state[3], state[7], state[11]
}

// invShiftRows performs the inverse ShiftRows operation on the state
func invShiftRows(state []byte) {
	// Row 0: no shift
	// Row 1: shift right by 1
	state[1], state[5], state[9], state[13] = state[13], state[1], state[5], state[9]
	// Row 2: shift right by 2
	state[2], state[6], state[10], state[14] = state[10], state[14], state[2], state[6]
	// Row 3: shift right by 3
	state[3], state[7], state[11], state[15] = state[7], state[11], state[15], state[3]
}

// mixColumns performs the MixColumns operation on the state
func mixColumns(state []byte) {
	for i := 0; i < 16; i += 4 {
		s0 := state[i]
		s1 := state[i+1]
		s2 := state[i+2]
		s3 := state[i+3]

		state[i] = gmul2[s0] ^ gmul3[s1] ^ s2 ^ s3
		state[i+1] = s0 ^ gmul2[s1] ^ gmul3[s2] ^ s3
		state[i+2] = s0 ^ s1 ^ gmul2[s2] ^ gmul3[s3]
		state[i+3] = gmul3[s0] ^ s1 ^ s2 ^ gmul2[s3]
	}
}

// invMixColumns performs the inverse MixColumns operation on the state
func invMixColumns(state []byte) {
	for i := 0; i < 16; i += 4 {
		s0 := state[i]
		s1 := state[i+1]
		s2 := state[i+2]
		s3 := state[i+3]

		state[i] = gmul(0x0e, s0) ^ gmul(0x0b, s1) ^ gmul(0x0d, s2) ^ gmul(0x09, s3)
		state[i+1] = gmul(0x09, s0) ^ gmul(0x0e, s1) ^ gmul(0x0b, s2) ^ gmul(0x0d, s3)
		state[i+2] = gmul(0x0d, s0) ^ gmul(0x09, s1) ^ gmul(0x0e, s2) ^ gmul(0x0b, s3)
		state[i+3] = gmul(0x0b, s0) ^ gmul(0x0d, s1) ^ gmul(0x09, s2) ^ gmul(0x0e, s3)
	}
}

// expandKey expands the 16-byte key into 11 round keys using AES key expansion
func expandKey(key []byte) [][]byte {
	if len(key) != 16 {
		panic("key must be 16 bytes")
	}

	// Pre-allocate all memory at once
	roundKeys := make([][]byte, 11)
	allKeys := make([]byte, 11*16)
	for i := range roundKeys {
		roundKeys[i] = allKeys[i*16 : (i+1)*16]
	}

	// Copy initial key
	copy(roundKeys[0], key)

	// Single temporary buffer for all rounds
	temp := make([]byte, 4)

	for i := 1; i < 11; i++ {
		prevKey := roundKeys[i-1]
		currKey := roundKeys[i]

		// First word
		// RotWord
		temp[0] = prevKey[13]
		temp[1] = prevKey[14]
		temp[2] = prevKey[15]
		temp[3] = prevKey[12]
		// SubWord
		temp[0] = sbox[temp[0]]
		temp[1] = sbox[temp[1]]
		temp[2] = sbox[temp[2]]
		temp[3] = sbox[temp[3]]
		// XOR with Rcon
		temp[0] ^= rcon[i-1]
		// XOR with previous key
		currKey[0] = prevKey[0] ^ temp[0]
		currKey[1] = prevKey[1] ^ temp[1]
		currKey[2] = prevKey[2] ^ temp[2]
		currKey[3] = prevKey[3] ^ temp[3]

		// Remaining words
		currKey[4] = currKey[0] ^ prevKey[4]
		currKey[5] = currKey[1] ^ prevKey[5]
		currKey[6] = currKey[2] ^ prevKey[6]
		currKey[7] = currKey[3] ^ prevKey[7]

		currKey[8] = currKey[4] ^ prevKey[8]
		currKey[9] = currKey[5] ^ prevKey[9]
		currKey[10] = currKey[6] ^ prevKey[10]
		currKey[11] = currKey[7] ^ prevKey[11]

		currKey[12] = currKey[8] ^ prevKey[12]
		currKey[13] = currKey[9] ^ prevKey[13]
		currKey[14] = currKey[10] ^ prevKey[14]
		currKey[15] = currKey[11] ^ prevKey[15]
	}

	return roundKeys
}

// padTweak pads an 8-byte tweak to 16 bytes according to KIASU-BC specification.
// The tweak is padded by placing each 2-byte pair at the start of a 4-byte group.
func padTweak(tweak []byte) []byte {
	if len(tweak) != 8 {
		panic("tweak must be 8 bytes")
	}
	padded := make([]byte, 16)
	for i := 0; i < 8; i += 2 {
		padded[i*2] = tweak[i]
		padded[i*2+1] = tweak[i+1]
	}
	return padded
}

// KiasuBCEncrypt encrypts a 16-byte block using KIASU-BC with the given key and tweak.
func KiasuBCEncrypt(key, tweak, block []byte) ([]byte, error) {
	if len(key) != 16 {
		return nil, errors.New("key must be 16 bytes")
	}
	if len(tweak) != 8 {
		return nil, errors.New("tweak must be 8 bytes")
	}
	if len(block) != 16 {
		return nil, errors.New("block must be 16 bytes")
	}

	// Expand key and pad tweak
	roundKeys := expandKey(key)
	paddedTweak := padTweak(tweak)

	// Create state
	state := make([]byte, 16)
	copy(state, block)

	// Initial round
	for i := 0; i < 16; i++ {
		state[i] ^= roundKeys[0][i] ^ paddedTweak[i]
	}

	// Main rounds
	for round := 1; round < 10; round++ {
		// SubBytes
		for i := 0; i < 16; i++ {
			state[i] = sbox[state[i]]
		}

		// ShiftRows
		shiftRows(state)

		// MixColumns
		mixColumns(state)

		// AddRoundKey
		for i := 0; i < 16; i++ {
			state[i] ^= roundKeys[round][i] ^ paddedTweak[i]
		}
	}

	// Final round
	// SubBytes
	for i := 0; i < 16; i++ {
		state[i] = sbox[state[i]]
	}

	// ShiftRows
	shiftRows(state)

	// AddRoundKey
	for i := 0; i < 16; i++ {
		state[i] ^= roundKeys[10][i] ^ paddedTweak[i]
	}

	return state, nil
}

// KiasuBCDecrypt decrypts a 16-byte block using KIASU-BC with the given key and tweak.
func KiasuBCDecrypt(key, tweak, block []byte) ([]byte, error) {
	if len(key) != 16 {
		return nil, errors.New("key must be 16 bytes")
	}
	if len(tweak) != 8 {
		return nil, errors.New("tweak must be 8 bytes")
	}
	if len(block) != 16 {
		return nil, errors.New("block must be 16 bytes")
	}

	// Expand key and pad tweak
	roundKeys := expandKey(key)
	paddedTweak := padTweak(tweak)

	// Create state
	state := make([]byte, 16)
	copy(state, block)

	// Initial round
	for i := 0; i < 16; i++ {
		state[i] ^= roundKeys[10][i] ^ paddedTweak[i]
	}

	// Inverse ShiftRows
	invShiftRows(state)

	// Inverse SubBytes
	for i := 0; i < 16; i++ {
		state[i] = invSbox[state[i]]
	}

	// Main rounds
	for round := 9; round > 0; round-- {
		// AddRoundKey
		for i := 0; i < 16; i++ {
			state[i] ^= roundKeys[round][i] ^ paddedTweak[i]
		}

		// Inverse MixColumns
		invMixColumns(state)

		// Inverse ShiftRows
		invShiftRows(state)

		// Inverse SubBytes
		for i := 0; i < 16; i++ {
			state[i] = invSbox[state[i]]
		}
	}

	// Final round
	// AddRoundKey
	for i := 0; i < 16; i++ {
		state[i] ^= roundKeys[0][i] ^ paddedTweak[i]
	}

	return state, nil
}
